/**
 * OWASP AppSensor
 * 
 * This file is part of the Open Web Application Security Project (OWASP)
 * AppSensor project. For details, please see
 * <a href="http://www.owasp.org/index.php/Category:OWASP_AppSensor_Project">
 * 	http://www.owasp.org/index.php/Category:OWASP_AppSensor_Project</a>.
 *
 * Copyright (c) 2010 - The OWASP Foundation
 * 
 * AppSensor is published by OWASP under the BSD license. You should read and accept the
 * LICENSE before you use, modify, and/or redistribute this software.
 * 
 * @author Michael Coates <a href="http://www.aspectsecurity.com">Aspect Security</a>
 * @author John Melton <a href="http://www.jtmelton.com/">jtmelton</a>
 * @created 2010
 */
namespace org.owasp.appsensor.errors
{

    using org.owasp.appsensor;
    using Owasp.Esapi;
    using System;
    using System.Text;

    /**
     * This class is the standard security exception for all AppSensor exceptions (except system exceptions).
     * Rather than have an exception per detection point, this exception is used 
     * along with a member variable holding the detection point event code, 
     * ie. RE1, SE2, ACE5, etc.
     * <p>
     * This allows applications to have a fairly simple mechanism to trigger
     * intrusion detection with appropriate handling for each event code.
     * (This assumes the AppSensorIntrusionDetector is used.)
     * <p>
     * To inform the IntrusionDetector that an AppSensor exception has occurred, 
     * you simply need to add code similar to the below at the appropriate point in code: 
     * <code>
     * 	new AppSensorException("RE1", "Unexpected Error", "Unexpected HTTP Commands were sent to the application.");
     * </code>
     * 
     * @author Michael Coates (michael.coates .at. owasp.org) 
     *         <a href="http://www.aspectsecurity.com">Aspect Security</a>
     * @author John Melton (jtmelton .at. gmail.com)
     *         <a href="http://www.jtmelton.com/">jtmelton</a>
     * @since February 24, 2010
     * @see org.owasp.esapi.errors.EnterpriseSecurityException
     */
    public class AppSensorException : Exception
    {

        /** Environment specific newline for writing out toString() */
        public String NEWLINE = Environment.NewLine;

        /** The Constant serialVersionUID. */
        private const long serialVersionUID = 1L;

        /** The logger. */
        protected ASLogger logger = APPSENSOR.AsUtilities.GetLogger("AppSensorException");

        /** The detailed log message to be sent to the log file, not to be sent to the user */
        protected String logMessage = null;

        /** The unique event code for appsensor, eg RE1, ACE3, SE2, etc. */
        private String eventCode;

        /**
         * Instantiates a new intrusion exception specific to appsensor.
         * 
         * @param eventcode
         * 			the appsensor event code, ie. RE1, SE2, ACE5, etc.
         * @param userMessage
         * 			the message to display to users
         * @param logMessage
         * 			the message logged
         */
        public AppSensorException(String eventCode, String userMessage, String logMessage)
            : base(userMessage)
        {
            this.logMessage = logMessage;
            this.eventCode = eventCode;
            logger.Error("AppSensorIntrusion - " + logMessage);
            Esapi.IntrusionDetector.AddException(this);
        }

        /**
         * Returns message meant for display to users
         *
         * Note that if you are unsure of what set this message, it would probably be a good idea to encode this message
         * before displaying it to the end user.
         * 
         * @return a String containing a message that is safe to display to users
         */
        public String UserMessage
        {
            get
            {
                return Message;
            }
        }

        /**
         * Returns a message that is safe to display in logs, but probably not to users
         * 
         * @return a String containing a message that is safe to display in logs, but probably not to users
         */
        public String LogMessage
        {
            get
            {
                return logMessage;
            }
        }

        /**
         * This method returns the AppSensor specific event code, such as RE1, SE2, ACE5, etc.
         * @return AppSensor specific event code, such as RE1, SE2, ACE5, etc.
         */
        public String EventCode
        {
            get
            {
                return this.eventCode;
            }
        }

        /**
         * Just an override of toString() for nice printing.
         */
        public override String ToString()
        {
            StringBuilder sb = new StringBuilder();
            sb.Append("Event code: ").Append(eventCode).Append(NEWLINE)
            .Append("User Message: ").Append(UserMessage).Append(NEWLINE)
            .Append("Log Message: ").Append(LogMessage).Append(NEWLINE);
            return sb.ToString();
        }

    }
}